using System;
using System.Collections.Generic;
using System.ComponentModel.DataAnnotations;
using System.Linq;
using System.Web.Mvc;
using HIPS.Web.UI.ViewModels.Shared;

namespace HIPS.Web.UI.ViewModels.HpiiSearch
{
    /// <summary>
    ///     View Model for the HpiiSearchController.SearchById action.
    /// </summary>
    public class SearchByIdViewModel : IValidatableObject
    {
        #region Properties

        [Required]
        [Display(Name = "Organisation (HPI-O)")]
        public string Hpio { get; set; }

        [StringLength(16, MinimumLength = 16, ErrorMessage = "HPI-I must be 16 characters long.")]
        [Display(Name = "HPI-I")]
        public string Hpii { get; set; }

        [StringLength(20, ErrorMessage = "Registration ID must be less than 20 characters long.")]
        [Display(Name = "Registration ID (AHPRA)")]
        public string RegistrationId { get; set; }

        [Required, StringLength(40, ErrorMessage = "Family Name must be less than 40 characters long.")]
        [Display(Name = "Family Name")]
        public string FamilyName { get; set; }

        [StringLength(100, ErrorMessage = "Given Name(s) must be less than 100 characters long.")]
        [Display(Name = "Given Name(s)")]
        public string GivenNames { get; set; }

        [Required]
        [Display(Name = "Sex")]
        public string SexCode { get; set; }

        [Display(Name = "Date of Birth"), DisplayFormat(DataFormatString = "dd/MM/yyyy")]
        public DateTime? DateOfBirth { get; set; }

        [Display(Name = "State")]
        public string StateCode { get; set; }

        [Display(Name = "Postcode")]
        [StringLength(4, ErrorMessage = "Postcode must be less than 4 characters long.")]
        public string Postcode { get; set; }

        public ResponseMessages ResponseMessages { get; internal set; }

        #endregion

        #region Data Source Properties

        public IEnumerable<SelectListItem> Hpios { get; internal set; }
        public IEnumerable<SelectListItem> Sexes { get; internal set; }
        public IEnumerable<SelectListItem> States { get; internal set; }

        #endregion

        /// <summary>
        ///     Initialises a new SearchByIdViewModel without reference data.
        /// </summary>
        /// <remarks>
        ///     Called automatically by MVC Model Binding.
        ///     Should initialise collections and complex objects to default/empty values.
        /// </remarks>
        public SearchByIdViewModel()
        {
            ResponseMessages = new ResponseMessages();
            Hpios = Enumerable.Empty<SelectListItem>();
            Sexes = Enumerable.Empty<SelectListItem>();
            States = Enumerable.Empty<SelectListItem>();
        }

        /// <summary>
        ///     Determines whether the specified object is valid.
        /// </summary>
        /// <returns>
        ///     A collection that holds failed-validation information.
        /// </returns>
        /// <param name="validationContext">The validation context.</param>
        /// <remarks>
        ///     Executed server-side after all annotation validators are successful.
        /// </remarks>
        public IEnumerable<ValidationResult> Validate(ValidationContext validationContext)
        {
            List<ValidationResult> errors = new List<ValidationResult>();

            if (String.IsNullOrWhiteSpace(Hpii) && String.IsNullOrWhiteSpace(RegistrationId))
            {
                errors.Add(new ValidationResult("Either the HPI-I Number or the Registration ID are required to complete an identifier search."));
            }

            return errors;
        }
    }
}